/***********************************************************
 * $Id$
 * 
 * FancyMail standalone OSGi server of the clazzes.org project
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.fancymail.server.dao.jdbc;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.clazzes.fancymail.server.dao.EMailRecipientDAO;
import org.clazzes.fancymail.server.entities.EMail;
import org.clazzes.fancymail.server.entities.EMailRecipient;
import org.clazzes.util.aop.jdbc.JdbcPreparedStatementAction;
import org.clazzes.util.sql.criteria.SQLCondition;
import org.clazzes.util.sql.criteria.SQLValue;
import org.clazzes.util.sql.dao.AbstrIdDAO;
import org.clazzes.util.sql.dao.StatementPreparer;
import org.clazzes.util.sql.helper.JDBCHelper;

/**
 * DAO implementation for {@link EMailRecipient} entities.
 */
public class JdbcEMailRecipientDAO extends AbstrIdDAO<EMailRecipient> implements
        EMailRecipientDAO {

    public JdbcEMailRecipientDAO() {
        super(EMailRecipient.class,"id","EMAIL_RECIPIENT",new String[]{
            "ID",
            "EMAIL_ID",
            "ENTITLEMENT",
            "ADDRESS",
            "PERSONAL_NAME"
        });
    }


    /* (non-Javadoc)
     * @see org.clazzes.util.sql.dao.AbstrBasicDAO#fillDtoFromResultSet(java.sql.ResultSet)
     */
    @Override
    protected EMailRecipient fillDtoFromResultSet(ResultSet rs)
            throws SQLException {
        
        EMailRecipient ret = new EMailRecipient();
        
        ret.setId(JDBCHelper.getLong(rs,1));
        ret.setEmailId(JDBCHelper.getLong(rs,2));
        ret.setEntitlement(rs.getInt(3));
        ret.setAddress(rs.getString(4));
        ret.setPersonalName(rs.getString(5));
         
        return ret;
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sql.dao.AbstrBasicDAO#fillPreparedStatementFromDto(java.sql.PreparedStatement, java.lang.Object)
     */
    @Override
    protected void fillPreparedStatementFromDto(PreparedStatement statement,
            EMailRecipient dto) throws SQLException {
        
        JDBCHelper.setLong(statement,1,dto.getId());
        JDBCHelper.setLong(statement,2,dto.getEmailId());
        statement.setInt(3,dto.getEntitlement());
        statement.setString(4,dto.getAddress());
        statement.setString(5,dto.getPersonalName());
    }


    @Override
    public void fetchAllForEMails(Collection<EMail> emails,
            SQLCondition condition, final StatementPreparer preparer) {
        
        final Map<Long,EMail> emailsById = new HashMap<Long,EMail>(emails.size());
        
        for (EMail email : emails) {
            emailsById.put(email.getId(),email);
        }
        
        String sql = this.getGenerator().innerJoin(
                this.getTableName(),
                "EMAIL",
                SQLCondition.eq(
                        SQLValue.tableColumn(this.getTableName(),"EMAIL_ID"),
                        SQLValue.tableColumn("EMAIL","ID")),
                condition,
                SQLValue.columnList(this.getTableName(),this.getColumnNames()));
        
        this.performWithPreparedStatement(sql,new JdbcPreparedStatementAction<Void>() {

            @Override
            public Void perform(PreparedStatement statement) throws Exception {
               
                preparer.fillInsertValues(statement);
                
                ResultSet rs = statement.executeQuery();
                
                while (rs.next()) {
                    
                    EMailRecipient recipient = JdbcEMailRecipientDAO.this.fillDtoFromResultSet(rs);
                    
                    EMail email = emailsById.get(recipient.getEmailId());
                    
                    if (email != null) {
                        email.addRecipient(recipient);
                    }
                }
                
                return null;
            }
        });
        
    }

    @Override
    public int[] deleteAllForEMails(final List<Long> emailIds) {
        
        String sql = this.getGenerator().delete(this.getTableName(),
                SQLCondition.eq(SQLValue.tableColumn(this.getTableName(),"EMAIL_ID"),
                        SQLValue.INSERT_VALUE));
        
        return this.performWithPreparedStatement(sql,new JdbcPreparedStatementAction<int[]>() {

            @Override
            public int[] perform(PreparedStatement statement) throws Exception {
               
                for (Long emailId : emailIds) {
                
                    JDBCHelper.setLong(statement,1,emailId);
                    statement.addBatch();
                }
                
                return statement.executeBatch();
            }
        });
    }

}
