/***********************************************************
 * $Id$
 *
 * FancyMail standalone OSGi server of the clazzes.org project
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.fancymail.server.dao.jdbc;

import java.io.Serializable;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.clazzes.fancymail.server.dao.EMailDAO;
import org.clazzes.fancymail.server.dao.EMailExtender;
import org.clazzes.fancymail.server.dao.EMailRecipientDAO;
import org.clazzes.fancymail.server.dao.EMailSenderDAO;
import org.clazzes.fancymail.server.entities.EMail;
import org.clazzes.fancymail.server.entities.EMailRecipient;
import org.clazzes.fancymail.server.entities.EMailSender;
import org.clazzes.util.aop.ThreadLocalManager;
import org.clazzes.util.aop.jdbc.JdbcPreparedStatementAction;
import org.clazzes.util.sql.criteria.SQLCondition;
import org.clazzes.util.sql.criteria.SQLValue;
import org.clazzes.util.sql.dao.AbstrIdDAO;
import org.clazzes.util.sql.dao.StatementPreparer;
import org.clazzes.util.sql.helper.JDBCHelper;

/**
 * DAO implementation for {@link EMailSender} entities.
 */
public class JdbcEMailDAO extends AbstrIdDAO<EMail> implements
        EMailDAO {

    private EMailRecipientDAO recipientDAO;
    private EMailSenderDAO senderDAO;

    public JdbcEMailDAO() {
        super(EMail.class,"id","EMAIL",new String[]{
            "ID",
            "SENDER_ID",
            "CREATED",
            "SENT",
            "SUBJECT",
            "BODY",
            "STATUS",
            "ERROR_COUNT",
            "LAST_ERROR_TEXT",
            "SENDING",
            "LAST_ERROR_EXCEPTION"
        });
    }

    @Override
    public EMail get(Serializable id) {

        Map<Long,EMail> emailCache =
                ThreadLocalManager.getBoundResource(EMailSenderCacheInterceptor.EMAIL_CACHE_KEY);

        if (emailCache != null) {

            EMail ret = emailCache.get(id);

            if (ret != null) {
                return ret;
            }
        }

        return super.get(id);
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sql.dao.AbstrBasicDAO#fillDtoFromResultSet(java.sql.ResultSet)
     */
    @Override
    protected EMail fillDtoFromResultSet(ResultSet rs)
            throws SQLException {

        EMail ret = new EMail();

        ret.setId(JDBCHelper.getLong(rs,1));
        ret.setSender(this.senderDAO.get(JDBCHelper.getLong(rs,2)));
        ret.setCreated(rs.getTimestamp(3));
        ret.setSentAt(rs.getTimestamp(4));
        ret.setSubject(rs.getString(5));
        ret.setBody(rs.getString(6));
        ret.setStatus(rs.getInt(7));
        ret.setErrorCount(rs.getInt(8));
        ret.setLastErrorText(rs.getString(9));
        ret.setSending(rs.getTimestamp(10));
        ret.setLastErrorException(rs.getString(11));

        Map<Long,EMail> emailCache =
                ThreadLocalManager.getBoundResource(EMailSenderCacheInterceptor.EMAIL_CACHE_KEY);

        if (emailCache != null) {
            emailCache.put(ret.getId(),ret);
        }

        return ret;
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sql.dao.AbstrBasicDAO#fillPreparedStatementFromDto(java.sql.PreparedStatement, java.lang.Object)
     */
    @Override
    protected void fillPreparedStatementFromDto(PreparedStatement statement,
            EMail dto) throws SQLException {

        JDBCHelper.setLong(statement,1,dto.getId());
        JDBCHelper.setLong(statement,2,dto.getSender() == null ? null : dto.getSender().getId());
        statement.setTimestamp(3,dto.getCreated() == null ? null : new Timestamp(dto.getCreated().getTime()));
        statement.setTimestamp(4,dto.getSentAt() == null ? null : new Timestamp(dto.getSentAt().getTime()));
        statement.setString(5,dto.getSubject());
        statement.setString(6,dto.getBody());
        statement.setInt(7,dto.getStatus());
        statement.setInt(8,dto.getErrorCount());
        statement.setString(9,dto.getLastErrorText());
        statement.setTimestamp(10,dto.getSending() == null ? null : new Timestamp(dto.getSending().getTime()));
        statement.setString(11,dto.getLastErrorException());
    }

    @Override
    public EMail save(EMail dto) {

        EMail ret = super.save(dto);

        if (ret.getRecipients() != null) {

            for (EMailRecipient r : ret.getRecipients()) {
                r.setEmailId(ret.getId());
            }

            this.recipientDAO.saveBatch(ret.getRecipients());
        }

        return ret;
    }

    protected SQLValue createdColumn() {
        return SQLValue.tableColumn(this.getTableName(),"CREATED");
    }

    protected SQLCondition createdFrom() {
        return SQLCondition.gtEquals(this.createdColumn(),SQLValue.INSERT_VALUE);
    }

    protected SQLCondition createdTo() {
        return SQLCondition.ltEquals(this.createdColumn(),SQLValue.INSERT_VALUE);
    }

    protected List<EMail> getEMailList(SQLCondition condition, StatementPreparer preparer) {

        List<EMail> emails = this.getListWithCondition(condition,preparer);

        this.recipientDAO.fetchAllForEMails(emails,condition,preparer);
        return emails;
    }

    protected List<EMail> getEMailListForUpdate(SQLCondition condition, StatementPreparer preparer) {

        List<EMail> emails = this.getListWithConditionForUpdate(condition,preparer);

        this.recipientDAO.fetchAllForEMails(emails,condition,preparer);
        return emails;
    }

    @Override
    public List<EMail> getAllInRange(final Date from, final Date to) {

        SQLCondition cond;

        if (from == null) {

            if (to == null) {
                cond = null;
            }
            else {
                cond = this.createdTo();
            }
        }
        else {

            if (to == null) {
                cond = this.createdFrom();
            }
            else {
                cond = SQLCondition.and(this.createdFrom(),this.createdTo());
            }
        }

        StatementPreparer preparer = new StatementPreparer() {

            @Override
            public void fillInsertValues(PreparedStatement statement)
                    throws SQLException {
                int i = 0;
                if (from != null) {
                    statement.setDate(++i,new java.sql.Date(from.getTime()));
                }
                if (to != null) {
                    statement.setDate(++i,new java.sql.Date(to.getTime()));
                }
            }
        };

        return this.getEMailList(cond,preparer);
    }

    @Override
    public List<EMail> getAllForRecipient(final String recipient, final Date from, final Date to) {

        SQLCondition cond = SQLCondition.eq(SQLValue.tableColumn("EMAIL_RECIPIENT","ADDRESS"),SQLValue.INSERT_VALUE);

        if (from == null) {

            if (to != null) {
                cond = SQLCondition.and(cond,this.createdTo());
            }
        }
        else {

            if (to == null) {
                cond = SQLCondition.and(cond,this.createdFrom());
            }
            else {
                cond = SQLCondition.and(cond,this.createdFrom(),this.createdTo());
            }
        }

        final StatementPreparer preparer = new StatementPreparer() {

            @Override
            public void fillInsertValues(PreparedStatement statement)
                    throws SQLException {
                int i = 0;
                statement.setString(++i,recipient);
                if (from != null) {
                    statement.setDate(++i,new java.sql.Date(from.getTime()));
                }
                if (to != null) {
                    statement.setDate(++i,new java.sql.Date(to.getTime()));
                }
            }
        };

        String sql = this.getGenerator().innerJoin(
                this.getTableName(),
                "EMAIL_RECIPIENT",
                SQLCondition.eq(
                        SQLValue.tableColumn("EMAIL_RECIPIENT","EMAIL_ID"),
                        SQLValue.tableColumn(this.getTableName(),"ID")),
                cond,
                SQLValue.columnList(this.getTableName(),this.getColumnNames()));

        List<EMail> ret = this.performWithPreparedStatement(sql,new JdbcPreparedStatementAction<List<EMail>>() {

            @Override
            public List<EMail> perform(PreparedStatement statement)
                    throws Exception {

                preparer.fillInsertValues(statement);

                ResultSet rs = statement.executeQuery();

                List<EMail> r = new ArrayList<EMail>();

                while (rs.next()) {
                    r.add(JdbcEMailDAO.this.fillDtoFromResultSet(rs));
                }

                return r;
            }

        });

        this.recipientDAO.fetchAllForEMails(ret,cond,preparer);

        return ret;
    }

    @Override
    public List<EMail> getAllForSender(String sender, final Date from, final Date to) {

        final EMailSender senderDTO = this.senderDAO.getByAddress(sender);

        if (senderDTO == null) {
            return null;
        }

        SQLCondition cond = SQLCondition.eq(SQLValue.tableColumn(this.getTableName(),"SENDER_ID"),SQLValue.INSERT_VALUE);

        if (from == null) {

            if (to != null) {
                cond = SQLCondition.and(cond,this.createdTo());
            }
        }
        else {

            if (to == null) {
                cond = SQLCondition.and(cond,this.createdFrom());
            }
            else {
                cond = SQLCondition.and(cond,this.createdFrom(),this.createdTo());
            }
        }

        final StatementPreparer preparer = new StatementPreparer() {

            @Override
            public void fillInsertValues(PreparedStatement statement)
                    throws SQLException {
                int i = 0;
                JDBCHelper.setLong(statement,++i,senderDTO.getId());
                if (from != null) {
                    statement.setDate(++i,new java.sql.Date(from.getTime()));
                }
                if (to != null) {
                    statement.setDate(++i,new java.sql.Date(to.getTime()));
                }
            }
        };

        return this.getEMailList(cond,preparer);
    }

    @Override
    public List<EMail> getAllByStatus(final int status, final int maxCount, EMailExtender extender) {

        SQLCondition cond = SQLCondition.eq(SQLValue.tableColumn(this.getTableName(),"STATUS"),SQLValue.INSERT_VALUE);

        final StatementPreparer preparer = new StatementPreparer() {

            @Override
            public void fillInsertValues(PreparedStatement statement)
                    throws SQLException {
                statement.setInt(1,status);
                statement.setMaxRows(maxCount);
            }
        };

        List<EMail> ret = this.getEMailList(cond,preparer);

        if (ret != null && ret.size() > 0 && extender != null) {
            extender.amendEMails(cond,preparer);
        }

        return ret;
    }

    @Override
    public List<EMail> getAllByStatusForUpdate(int status, int maxCount,
            EMailExtender extender) {

        SQLCondition cond = SQLCondition.eq(SQLValue.tableColumn(this.getTableName(),"STATUS"),SQLValue.INSERT_VALUE);

        final StatementPreparer preparer = new StatementPreparer() {

            @Override
            public void fillInsertValues(PreparedStatement statement)
                    throws SQLException {
                statement.setInt(1,status);
                statement.setMaxRows(maxCount);
            }
        };

        List<EMail> ret = this.getEMailListForUpdate(cond,preparer);

        if (ret != null && ret.size() > 0 && extender != null) {
            extender.amendEMails(cond,preparer);
        }

        return ret;
    }

    @Override
    public List<EMail> getStaleSending(Date before, int maxCount,
            EMailExtender extender) {

        SQLCondition cond =
            SQLCondition.and(
                SQLCondition.eq(SQLValue.tableColumn(this.getTableName(),"STATUS"),SQLValue.integerValue(EMail.MAIL_STATUS_SENDING)),
                SQLCondition.ltEquals(SQLValue.tableColumn(this.getTableName(),"SENDING"),SQLValue.INSERT_VALUE));

        final StatementPreparer preparer = new StatementPreparer() {

            @Override
            public void fillInsertValues(PreparedStatement statement)
                    throws SQLException {
                statement.setTimestamp(1,new java.sql.Timestamp(before.getTime()));
                statement.setMaxRows(maxCount);
            }
        };

        List<EMail> ret = this.getEMailListForUpdate(cond,preparer);

        if (ret != null && ret.size() > 0 && extender != null) {
            extender.amendEMails(cond,preparer);
        }

        return ret;
    }

    @Override
    public List<EMail> getAllForSenderRecipient(String sender,
            final String recipient, final Date from, final Date to) {

        final EMailSender senderDTO = this.senderDAO.getByAddress(sender);

        if (senderDTO == null) {
            return null;
        }

        SQLCondition cond;
        SQLCondition cond1 = SQLCondition.eq(SQLValue.tableColumn(this.getTableName(),"SENDER_ID"),SQLValue.INSERT_VALUE);
        SQLCondition cond2 = SQLCondition.eq(SQLValue.tableColumn("EMAIL_RECIPIENT","ADDRESS"),SQLValue.INSERT_VALUE);

        if (from == null) {

            if (to == null) {
                cond = SQLCondition.and(cond1,cond2);
            }
            else {
                cond = SQLCondition.and(cond1,cond2,this.createdTo());
            }
        }
        else {

            if (to == null) {
                cond = SQLCondition.and(cond1,cond2,this.createdFrom());
            }
            else {
                cond = SQLCondition.and(cond1,cond2,this.createdFrom(),this.createdTo());
            }
        }

        final StatementPreparer preparer = new StatementPreparer() {

            @Override
            public void fillInsertValues(PreparedStatement statement)
                    throws SQLException {
                int i = 0;
                JDBCHelper.setLong(statement,++i,senderDTO.getId());
                statement.setString(++i,recipient);
                if (from != null) {
                    statement.setDate(++i,new java.sql.Date(from.getTime()));
                }
                if (to != null) {
                    statement.setDate(++i,new java.sql.Date(to.getTime()));
                }
            }
        };

        String sql = this.getGenerator().innerJoin(
                this.getTableName(),
                "EMAIL_RECIPIENT",
                SQLCondition.eq(
                        SQLValue.tableColumn("EMAIL_RECIPIENT","EMAIL_ID"),
                        SQLValue.tableColumn(this.getTableName(),"ID")),
                cond,
                SQLValue.columnList(this.getTableName(),this.getColumnNames()));

        List<EMail> ret = this.performWithPreparedStatement(sql,new JdbcPreparedStatementAction<List<EMail>>() {

            @Override
            public List<EMail> perform(PreparedStatement statement)
                    throws Exception {

                preparer.fillInsertValues(statement);

                ResultSet rs = statement.executeQuery();

                List<EMail> r = new ArrayList<EMail>();

                while (rs.next()) {
                    r.add(JdbcEMailDAO.this.fillDtoFromResultSet(rs));
                }

                return r;
            }

        });

        this.recipientDAO.fetchAllForEMails(ret,cond,preparer);

        return ret;
    }

    @Override
    public List<Long> getOutdatedIds(final Date to) {

        String sql = this.getGenerator().select(this.getTableName(),
                new SQLValue[]{SQLValue.tableColumn(this.getTableName(),"ID")},this.createdTo());

        List<Long> ids =
        this.performWithPreparedStatement(sql,new JdbcPreparedStatementAction<List<Long>>() {

            @Override
            public List<Long> perform(PreparedStatement statement) throws Exception {

                statement.setDate(1,new java.sql.Date(to.getTime()));

                ResultSet rs = statement.executeQuery();

                List<Long> r = new ArrayList<Long>();

                while (rs.next()) {
                    r.add(JDBCHelper.getLong(rs,1));
                }
                return r;
            }
        });

        return ids;
    }

    @Override
    public void deleteByIds(List<Long> ids) {

        this.recipientDAO.deleteAllForEMails(ids);
        this.deleteBatch(ids);
    }

    public void setRecipientDAO(EMailRecipientDAO recipientDAO) {
        this.recipientDAO = recipientDAO;
    }

    public void setSenderDAO(EMailSenderDAO senderDAO) {
        this.senderDAO = senderDAO;
    }
}
