/***********************************************************
 * $Id: JdbcEMailDAO.java 212 2013-10-04 11:55:31Z fancymail $
 * 
 * FancyMail standalone OSGi server of the clazzes.org project
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.fancymail.server.dao.jdbc;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import org.clazzes.fancymail.server.dao.EMailAttachmentDAO;
import org.clazzes.fancymail.server.dao.EMailDAO;
import org.clazzes.fancymail.server.entities.EMail;
import org.clazzes.fancymail.server.entities.EMailAttachment;
import org.clazzes.fancymail.server.entities.EMailSender;
import org.clazzes.util.aop.jdbc.JdbcPreparedStatementAction;
import org.clazzes.util.sql.criteria.SQLCondition;
import org.clazzes.util.sql.criteria.SQLValue;
import org.clazzes.util.sql.dao.AbstrIdDAO;
import org.clazzes.util.sql.dao.StatementPreparer;
import org.clazzes.util.sql.helper.JDBCHelper;

/**
 * DAO implementation for {@link EMailSender} entities.
 */
public class JdbcEMailAttachmentDAO extends AbstrIdDAO<EMailAttachment> implements
        EMailAttachmentDAO {

    private EMailDAO mailDAO;
    
    public JdbcEMailAttachmentDAO() {
        super(EMailAttachment.class,"id","EMAIL_ATTACHMENT",new String[]{
            "ID",
            "EMAIL_ID",
            "PRETTY_NAME",
            "MIME_TYPE",
            "CONTENT"
        });
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sql.dao.AbstrBasicDAO#fillDtoFromResultSet(java.sql.ResultSet)
     */
    @Override
    protected EMailAttachment fillDtoFromResultSet(ResultSet rs)
            throws SQLException {
        
        EMailAttachment ret = new EMailAttachment();
        
        ret.setId(JDBCHelper.getLong(rs,1));
        ret.setMail(this.mailDAO.get(JDBCHelper.getLong(rs,2)));
        ret.setPrettyName(rs.getString(3));
        ret.setMimeType(rs.getString(4));
        ret.setContent(rs.getBytes(5));

        return ret;
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sql.dao.AbstrBasicDAO#fillPreparedStatementFromDto(java.sql.PreparedStatement, java.lang.Object)
     */
    @Override
    protected void fillPreparedStatementFromDto(PreparedStatement statement,
            EMailAttachment dto) throws SQLException {
        
        JDBCHelper.setLong(statement,1,dto.getId());
        JDBCHelper.setLong(statement,2,dto.getEMail() == null ? null : ((EMail)dto.getEMail()).getId());
        statement.setString(3,dto.getPrettyName());
        statement.setString(4,dto.getMimeType());
        statement.setBytes(5,dto.getContent());
    }

    @Override
    public List<EMailAttachment> getAllForEMail(final Long emailId) {
        
        SQLCondition cond = SQLCondition.eq(SQLValue.tableColumn(this.getTableName(),"EMAIL_ID"),SQLValue.INSERT_VALUE);
        
        final StatementPreparer preparer = new StatementPreparer() {
            
            @Override
            public void fillInsertValues(PreparedStatement statement)
                    throws SQLException {
                JDBCHelper.setLong(statement,1,emailId);
            }
        };
        
        return this.getListWithCondition(cond,preparer);
    }

    @Override
    public void amendEMails(SQLCondition condition, final StatementPreparer preparer) {
        
        String sql = this.getGenerator().innerJoin(
                this.getTableName(),
                "EMAIL",
                SQLCondition.eq(
                        SQLValue.tableColumn(this.getTableName(),"EMAIL_ID"),
                        SQLValue.tableColumn("EMAIL","ID")),
                condition,
                SQLValue.columnList(this.getTableName(),this.getColumnNames()));
        
        this.performWithPreparedStatement(sql,new JdbcPreparedStatementAction<Void>() {

            @Override
            public Void perform(PreparedStatement statement) throws Exception {
               
                preparer.fillInsertValues(statement);
                
                ResultSet rs = statement.executeQuery();
                
                while (rs.next()) {
                    
                    EMailAttachment att = JdbcEMailAttachmentDAO.this.fillDtoFromResultSet(rs);

                    EMail mail = (EMail) att.getEMail();
                    mail.addAttachment(att);
                }
                
                return null;
            }
        });
    }

    @Override
    public void deleteAllForEMails(final List<Long> emailIds) {
        
        String sql = this.getGenerator().delete(this.getTableName(),
                SQLCondition.eq(SQLValue.tableColumn(this.getTableName(),"EMAIL_ID"),
                        SQLValue.INSERT_VALUE));
        
        this.performWithPreparedStatement(sql,new JdbcPreparedStatementAction<int[]>() {

            @Override
            public int[] perform(PreparedStatement statement) throws Exception {
               
                for (Long emailId : emailIds) {
                
                    JDBCHelper.setLong(statement,1,emailId);
                    statement.addBatch();
                }
                
                return statement.executeBatch();
            }
        });
    }

    public void setMailDAO(EMailDAO mailDAO) {
        this.mailDAO = mailDAO;
    }
}
