/***********************************************************
 * $Id$
 *
 * FancyMail standalone OSGi server of the clazzes.org project
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.fancymail.server.dao;

import java.util.Date;
import java.util.List;

import org.clazzes.fancymail.server.entities.EMail;
import org.clazzes.util.sql.dao.IIdDAO;

/**
 * @author wglas
 *
 */
public interface EMailDAO extends IIdDAO<EMail> {

    /**
     * @param from The date to find emails created after or <code>null</code>, if searching
     *              without lower bound.
     * @param to The date to find emails created before or <code>null</code>, if searching
     *              without upper bound.
     * @return All emails created before the given date <code>to</code> or
     *          after the given data <code>from</code>. The search interval is open at
     *          <code>from</code> and closed at <code>to</code>.
     */
    List<EMail> getAllInRange(Date from, Date to);

    /**
     * @param recipient The e-mail address of the recipient to search for.
     * @param from The date to find emails created after or <code>null</code>, if searching
     *              without lower bound.
     * @param to The date to find emails created before or <code>null</code>, if searching
     *              without upper bound.
      * @return All emails targeting at specified recipient.
     */
    List<EMail> getAllForRecipient(String recipient, Date from, Date to);

    /**
     * @param sender The e-mail address of the sender to search for.
     * @param from The date to find emails created after or <code>null</code>, if searching
     *              without lower bound.
     * @param to The date to find emails created before or <code>null</code>, if searching
     *              without upper bound.
      * @return All emails targeting at specified recipient.
     */
    List<EMail> getAllForSender(String sender, Date from, Date to);

   /**
     * Return up to <code>maxCount</code> emails to send.
     * @param status The status of the mail to retrieve.
     * @param maxCount The maximal number of mails to return.
     * @param extender An extender to use for inserting attachments to the mails.
     * @return A list of EMails with the given status.
     */
    List<EMail> getAllByStatus(int status, int maxCount, EMailExtender extender);

   /**
     * Return up to <code>maxCount</code> emails to send by <code>SELECT FOR UPDATE</code>.
     * @param status The status of the mail to retrieve.
     * @param maxCount The maximal number of mails to return.
     * @param extender An extender to use for inserting attachments to the mails.
     * @return A list of EMails with the given status.
     */
    List<EMail> getAllByStatusForUpdate(int status, int maxCount, EMailExtender extender);

    /**
     * <p>Return up to <code>maxCount</code> emails being in status
     * <code>SENDING</code> and started to be in this status before
     * the given point in time.</p>
     *
     * <p>This call uses <code>SELECT FOR UPDATE</code> in order to
     * </p>
     *
     * @param before The point in time before the returned mail should have been processed.
     * @param maxCount The maximal number of mails to return.
     * @param extender An extender to use for inserting attachments to the mails.
     * @return A list of e-mails, which which are in the sending stat since a point
     *         in time before the given timestamp.
     */
    List<EMail> getStaleSending(Date before, int maxCount, EMailExtender extender);

    /**
     * @param sender The e-mail address of the sender to search for.
     * @param recipient The e-mail address of the sender to search for.
     * @param from The date to find emails created after or <code>null</code>, if searching
     *              without lower bound.
     * @param to The date to find emails created before or <code>null</code>, if searching
     *              without upper bound.
      * @return All emails targeting at specified recipient.
     */
    List<EMail> getAllForSenderRecipient(String sender, String recipient,
            Date from, Date to);

    /**
     * Get the IDS of all mails generated before the given date.
     *
     * @param to The date to find emails created before.
     * @return The IDs of the the mails to be deleted.
     */
    public List<Long> getOutdatedIds(Date to);

    /**
     * Actually delete a list of mails togethe with the associated recipient records.
     *
     * @param ids The list of IDs of the mail and recipients to delete.
     */
    public void deleteByIds(List<Long> ids);
}
