package org.clazzes.fancymail.server;

import java.sql.SQLException;
import java.util.function.Consumer;

import javax.sql.DataSource;

import org.aopalliance.intercept.MethodInterceptor;
import org.clazzes.fancymail.mail.EMailException;
import org.clazzes.fancymail.sending.EMailEngine;
import org.clazzes.fancymail.server.api.FancyMailServerSMSService;
import org.clazzes.fancymail.server.api.FancyMailServerService;
import org.clazzes.fancymail.server.dao.jdbc.EMailSenderCacheInterceptor;
import org.clazzes.fancymail.server.dao.jdbc.EmailThreadInterceptor;
import org.clazzes.fancymail.server.dao.jdbc.JdbcEMailAttachmentDAO;
import org.clazzes.fancymail.server.dao.jdbc.JdbcEMailDAO;
import org.clazzes.fancymail.server.dao.jdbc.JdbcEMailRecipientDAO;
import org.clazzes.fancymail.server.dao.jdbc.JdbcEMailSenderDAO;
import org.clazzes.fancymail.server.dao.jdbc.JdbcSMSDAO;
import org.clazzes.fancymail.server.dao.jdbc.JdbcSMSDestinationDAO;
import org.clazzes.fancymail.server.dao.jdbc.JdbcSMSSenderDAO;
import org.clazzes.fancymail.server.dao.jdbc.SMSSenderCacheInterceptor;
import org.clazzes.fancymail.server.dao.jdbc.SMSThreadInterceptor;
import org.clazzes.fancymail.server.service.EMailService;
import org.clazzes.fancymail.server.service.SMSService;
import org.clazzes.fancymail.server.service.impl.DelegatingSMSChannel;
import org.clazzes.fancymail.server.service.impl.EMail2MimeMessageResolverImpl;
import org.clazzes.fancymail.server.service.impl.EMailServiceImpl;
import org.clazzes.fancymail.server.service.impl.FancyMailServerSMSServiceImpl;
import org.clazzes.fancymail.server.service.impl.FancyMailServerServiceImpl;
import org.clazzes.fancymail.server.service.impl.GarbageCollector;
import org.clazzes.fancymail.server.service.impl.SMSServiceImpl;
import org.clazzes.fancymail.server.service.impl.SchemaChecker;
import org.clazzes.fancymail.sms.JsmppSMSChannel;
import org.clazzes.fancymail.sms.SMSEngine;
import org.clazzes.fancymail.sms.SMSException;
import org.clazzes.jdbc2xml.schema.ISchemaEngineFactory;
import org.clazzes.jdbc2xml.schema.impl.SchemaEngineFactoryImpl;
import org.clazzes.svc.api.Component;
import org.clazzes.svc.api.ComponentManager;
import org.clazzes.svc.api.ComponentSupport;
import org.clazzes.svc.api.ConfigWrapper;
import org.clazzes.svc.api.ConfigurationEngine;
import org.clazzes.svc.api.ServiceContext;
import org.clazzes.svc.api.ServiceRegistry;
import org.clazzes.util.aop.DAOException;
import org.clazzes.util.aop.ProxyFactory;
import org.clazzes.util.aop.jdbc.JdbcTransactionInterceptor;
import org.clazzes.util.sql.SQLDialect;
import org.clazzes.util.sql.SQLStatementGenerator;
import org.clazzes.util.sql.dao.HiLoIdGenerator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FancymailServerComponent extends ComponentSupport implements Component {

    private static final Logger log = LoggerFactory.getLogger(FancymailServerComponent.class);

    public static final String CONFIG_PID = "org.clazzes.fancymail.server";
    public static final String DS_TL_KEY = "org.clazzes.fancymail.server::DataSource";

    protected final static class ConfigListener
                             extends ComponentSupport
                               implements Consumer<ConfigWrapper>, AutoCloseable {

        private final ServiceRegistry registry;
        private final ComponentManager componentManager;

        private EMailEngine mailEngine;
        private SMSEngine smsEngine;
        private GarbageCollector garbageCollector;

        public ConfigListener(ServiceRegistry registry, ComponentManager componentManager) {
            this.registry = registry;
            this.componentManager = componentManager;
        }

        @Override
        public void accept(ConfigWrapper config) {

            this.close();

            String datasourceName =
                config.getString("datasourceName","FANCYMAIL");

            String smtpHost = config.getString("smtpHost","localhost");
            int smtpPort = config.getInt("smtpPort",25);
            boolean smtpSSL = config.getBoolean("smtpSSL",false);

            String smtpUser = config.getString("smtpUser");
            String smtpPassword = config.getString("smtpPassword");

            int smtpTimeoutSecs = config.getInt("smtpTimeoutSecs",15);

            int sleepPerMailMillis = config.getInt("sleepPerMailMillis",1000);
            int pollIntervalMillis = config.getInt("pollIntervalMillis",60000);

            int maxRetries = config.getInt("maxRetries",15);
            int maxBulkSize = config.getInt("maxBulkSize",20);

            int staleTimeoutMinutes = config.getInt("staleTimeoutMinutes",15);
            int monthsToKeep = config.getInt("monthsToKeep",36);

            String gcTimeOfDay = config.getString("gcTimeOfDay","1:30");

            String smsUrl = config.getString("smsUrl");
            int smsPort = config.getInt("smsPort",5040);

            String smsUser = config.getString("smsUser");
            String smsPassword = config.getString("smsPassword");
            String smsSystemType = config.getString("smsSystemType");

            String smsBackend = config.getString("smsBackend","static:smpp");

            int smppTimeoutSecs = config.getInt("smppTimeoutSecs",15);

            this.addListener(this.registry.listen(datasourceName,DataSource.class,
                (ds) -> { synchronized(this) {


                    // 1. perform schema check.
                    ISchemaEngineFactory schemaEngineFactory = new SchemaEngineFactoryImpl();

                    SchemaChecker schemaChecker =
                        new SchemaChecker(schemaEngineFactory.newSchemaEngine());

                    schemaChecker.provideDatasource(ds);

                    // set up interceptors, ID generators,...
                    JdbcTransactionInterceptor jdbcInterceptor = new JdbcTransactionInterceptor();
                    jdbcInterceptor.setDataSource(ds);
                    jdbcInterceptor.setThreadLocalKey(DS_TL_KEY);

                    EMailSenderCacheInterceptor senderCacheInterceptor = new EMailSenderCacheInterceptor();
                    EmailThreadInterceptor emailThreadInterceptor = new EmailThreadInterceptor();

                    SMSSenderCacheInterceptor smsSenderCacheInterceptor = new SMSSenderCacheInterceptor();
                    SMSThreadInterceptor smsThreadInterceptor = new SMSThreadInterceptor();

                    ProxyFactory jdbcProxyFactory = new ProxyFactory();
                    jdbcProxyFactory.setInterceptors(new MethodInterceptor[] {
                        emailThreadInterceptor,
                        jdbcInterceptor,
                        senderCacheInterceptor
                    });

                    ProxyFactory smsJdbcProxyFactory = new ProxyFactory();
                    smsJdbcProxyFactory.setInterceptors(new MethodInterceptor[] {
                        smsThreadInterceptor,
                        jdbcInterceptor,
                        smsSenderCacheInterceptor
                    });

                    HiLoIdGenerator idGenerator = new HiLoIdGenerator();
                    idGenerator.setDataSource(ds);
                    try {
                        idGenerator.initialize();
                    } catch (SQLException e) {
                        throw new DAOException("Error initializing ID generator",e);
                    }

                    SQLStatementGenerator sqlGenerator = new SQLStatementGenerator(SQLDialect.ISO);

                    JdbcEMailSenderDAO emailSenderDAO = new JdbcEMailSenderDAO();
                    emailSenderDAO.setGenerator(sqlGenerator);
                    emailSenderDAO.setIdGenerator(idGenerator);
                    emailSenderDAO.setThreadLocalKey(DS_TL_KEY);

                    JdbcEMailRecipientDAO emailRecipientDAO = new JdbcEMailRecipientDAO();

                    emailRecipientDAO.setGenerator(sqlGenerator);
                    emailRecipientDAO.setIdGenerator(idGenerator);
                    emailRecipientDAO.setThreadLocalKey(DS_TL_KEY);

                    JdbcEMailDAO emailDAO = new JdbcEMailDAO();
                    emailDAO.setGenerator(sqlGenerator);
                    emailDAO.setIdGenerator(idGenerator);
                    emailDAO.setThreadLocalKey(DS_TL_KEY);
                    emailDAO.setSenderDAO(emailSenderDAO);
                    emailDAO.setRecipientDAO(emailRecipientDAO);

                    JdbcEMailAttachmentDAO emailAttachmentDAO = new JdbcEMailAttachmentDAO();
                    emailAttachmentDAO.setGenerator(sqlGenerator);
                    emailAttachmentDAO.setIdGenerator(idGenerator);
                    emailAttachmentDAO.setThreadLocalKey(DS_TL_KEY);
                    emailAttachmentDAO.setMailDAO(emailDAO);

                    EMailServiceImpl emailServiceImpl = new EMailServiceImpl();
                    emailServiceImpl.setMailAttachmentDAO(emailAttachmentDAO);
                    emailServiceImpl.setMailDAO(emailDAO);
                    emailServiceImpl.setMailSenderDAO(emailSenderDAO);
                    emailServiceImpl.setMaxBulkSize(maxBulkSize);
                    emailServiceImpl.setMaxRetries(maxRetries);
                    emailServiceImpl.setStaleTimeoutMinutes(staleTimeoutMinutes);

                    EMailService emailService = jdbcProxyFactory.getTypedProxy(emailServiceImpl,EMailService.class);

                    EMail2MimeMessageResolverImpl mailResolver = new EMail2MimeMessageResolverImpl();

                    try {
                        this.mailEngine = new EMailEngine();
                        this.mailEngine.setEMail2MimeMessageResolver(mailResolver);
                        this.mailEngine.setEMailFactory(emailService);
                        this.mailEngine.setEMailReportConsumer(emailService);
                        this.mailEngine.setSmtpPort(smtpPort);
                        this.mailEngine.setSmtpHost(smtpHost);
                        this.mailEngine.setSmtpSSL(smtpSSL);
                        this.mailEngine.setSmtpUser(smtpUser);
                        this.mailEngine.setSmtpPassword(smtpPassword);
                        this.mailEngine.setSmtpTimeoutSecs(smtpTimeoutSecs);
                        this.mailEngine.setPollIntervalMillis(pollIntervalMillis);
                        this.mailEngine.setSleepPerMailMillis(sleepPerMailMillis);

                    } catch (EMailException e) {
                        throw new DAOException("Error setting up EMailEngine",e);
                    }

                    FancyMailServerServiceImpl fancyMailServerService = new FancyMailServerServiceImpl();
                    fancyMailServerService.setMailService(emailService);

                    // SMS DAOs
                    JdbcSMSSenderDAO smsSenderDAO = new JdbcSMSSenderDAO();
                    smsSenderDAO.setGenerator(sqlGenerator);
                    smsSenderDAO.setIdGenerator(idGenerator);
                    smsSenderDAO.setThreadLocalKey(DS_TL_KEY);

                    JdbcSMSDestinationDAO smsRecipientDAO = new JdbcSMSDestinationDAO();
                    smsRecipientDAO.setGenerator(sqlGenerator);
                    smsRecipientDAO.setIdGenerator(idGenerator);
                    smsRecipientDAO.setThreadLocalKey(DS_TL_KEY);

                    JdbcSMSDAO smsDAO = new JdbcSMSDAO();
                    smsDAO.setGenerator(sqlGenerator);
                    smsDAO.setIdGenerator(idGenerator);
                    smsDAO.setThreadLocalKey(DS_TL_KEY);
                    smsDAO.setDestinationDAO(smsRecipientDAO);
                    smsDAO.setSenderDAO(smsSenderDAO);

                    SMSServiceImpl smsServiceImpl = new SMSServiceImpl();
                    smsServiceImpl.setMaxBulkSize(maxBulkSize);
                    smsServiceImpl.setMaxRetries(maxRetries);
                    smsServiceImpl.setSmsDAO(smsDAO);
                    smsServiceImpl.setSmsSenderDAO(smsSenderDAO);
                    smsServiceImpl.setStaleTimeoutMinutes(staleTimeoutMinutes);

                    SMSService smsService = smsJdbcProxyFactory.getTypedProxy(smsServiceImpl,SMSService.class);

                    JsmppSMSChannel smppChannel = new JsmppSMSChannel();
                    smppChannel.setUser(smsUser);
                    smppChannel.setPort(smsPort);
                    smppChannel.setUrl(smsUrl);
                    smppChannel.setSystemType(smsSystemType);
                    smppChannel.setPassword(smsPassword);
                    smppChannel.setTimeoutSecs(smppTimeoutSecs);

                    DelegatingSMSChannel smsChannnel = new DelegatingSMSChannel();
                    smsChannnel.setSmppDelegate(smppChannel);
                    smsChannnel.setSmsBackend(smsBackend);
                    smsChannnel.setSmsChannelMap(this.registry);

                    try {
                        this.smsEngine = new SMSEngine();
                        this.smsEngine.setSleepPerSMSMillis(sleepPerMailMillis);
                        this.smsEngine.setPollIntervalMillis(pollIntervalMillis);
                        this.smsEngine.setSMSChannel(smsChannnel);
                        this.smsEngine.setSMSReportConsumer(smsService);
                        this.smsEngine.setSMSFactory(smsService);

                    } catch (SMSException e) {
                        throw new DAOException("Error setting up SMSEngine",e);
                    }

                    FancyMailServerSMSServiceImpl fancyMailServerSMSService = new FancyMailServerSMSServiceImpl();
                    fancyMailServerSMSService.setSmsService(smsService);

                    this.garbageCollector = new GarbageCollector();
                    this.garbageCollector.setMailService(emailService);
                    this.garbageCollector.setMonthsToKeep(monthsToKeep);
                    this.garbageCollector.setTimeOfDay(gcTimeOfDay);
                    this.garbageCollector.setSmsService(smsService);

                    try {
                        this.mailEngine.init();
                        emailThreadInterceptor.setEMailEngine(this.mailEngine);
                        log.info("Successfully started mailEngine");
                        this.registry.addService("fancymail-server",FancyMailServerService.class,fancyMailServerService);
                    } catch (EMailException e) {
                        log.warn("Error starting mail engine",e);
                    }

                    try {
                        this.smsEngine.init();
                        smsThreadInterceptor.setSMSEngine(this.smsEngine);
                        log.info("Successfully started smsEngine");
                        this.registry.addService("fancymail-server",FancyMailServerSMSService.class,fancyMailServerSMSService);
                    } catch (SMSException e) {
                        log.warn("Error starting sms engine", e);
                    }

                    this.garbageCollector.start();

                    log.info("fancymail services are now up and running.");
                    this.componentManager.commit();
                }},
                (ds) -> {
                    this.closeBusinessLogics();
                })
            );
        }

        public synchronized void closeBusinessLogics() {


            if (this.garbageCollector != null) {
                this.garbageCollector.stop();
                this.garbageCollector = null;
            }

            if (this.mailEngine != null) {
                try {
                    this.mailEngine.uninit();
                    log.info("Successfully stopped mailEngine");
                } catch (Exception e) {
                    log.warn("Error stopping mail engine",e);
                }
                this.mailEngine = null;
            }

            if (this.smsEngine != null) {

                try {
                    this.smsEngine.uninit();
                    log.info("Successfully stopped smsEngine");
                } catch (Exception e) {
                    log.warn("Error stopping sms engine", e);
                }
                this.smsEngine = null;
            }

            log.info("fancymail services have been shut down.");
            this.removeAllServices(this.registry);
        }

        @Override
        public void close() {
            // close DB connection listener.
            this.closeAllListeners();
            // close all active services.
            this.closeBusinessLogics();
        }
    }

    @Override
    public void start(ServiceContext context) throws Exception {
        ServiceRegistry registry = context.getService(ServiceRegistry.class).get();
        ConfigurationEngine ce = context.getService(ConfigurationEngine.class).get();
        ComponentManager componentManager = context.getService(ComponentManager.class).get();

        ConfigListener cl = new ConfigListener(registry,componentManager);

        // this listener closes the business logics and DB listener.
        this.addListener(cl);
        // this listener closes the config interest.
        this.addListener(ce.listen(CONFIG_PID,cl));
    }

    @Override
    public void stop(ServiceContext context) throws Exception {

        this.closeAllListeners();
    }

}
