/***********************************************************
 * $Id$
 *
 * FancyMail standalone OSGi server of the clazzes.org project
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.fancymail.server.api;

import java.util.Collection;
import java.util.Date;
import java.util.List;

import javax.activation.DataSource;

/**
 * This interface is the OSGi service interface for external applications to
 * interfere with the fancymail server.
 */
public interface FancyMailServerService {

    /**
     * Search the given address in the list of configured senders.
     * @param emailAddr The RFC822 e-mail address to search for.
     * @return An {@link EMailSenderDTO} or <code>null</code>, if the given address
     *         has not yet been configured as a sender.
     */
    EMailSenderDTO getSenderAddress(String emailAddr);

    /**
     * @return All configured email senders.
     */
    List<EMailSenderDTO> getAllSenders();

    /**
     * Configure a new sender.
     * @param sender The new sender with at least a filled in address.
     */
    void insertSender(EMailSenderDTO sender);

    /**
     * Update the personal name and replyTo of a previously configured sender.
     * @param sender The updated sender with at least a filled in, previously configured sender
     *               address.
     */
    void updateSender(EMailSenderDTO sender);

    /**
     * Update the personal name and replyTo of a list of previously configured senders.
     * @param senders The list of updated senders with at least a filled in,
     *                previously configured sender address.
     */
    void updateSenders(Collection<EMailSenderDTO> senders);

    /**
     * Queue an email for delivery. The sender of the email must have been previously
     * configured using {@link #insertSender(EMailSenderDTO)}.
     * @param email The email to enqueue with at least on <code>To:</code> recipient.
     * @return The generated <code>created</code> date of the persisted e-mail.
     */
    Date queueMail(EMailDTO email);

    /**
     * Queue an email for delivery. The sender of the email must have been previously
     * configured using {@link #insertSender(EMailSenderDTO)}.
     * @param email The email to enqueue with at least on <code>To:</code> recipient.
     * @param attachments The list of attachments.
     * @return The generated <code>created</code> date of the persisted e-mail.
     */
    Date queueMailWithAttachments(EMailDTO email, List<DataSource> attachments);


    /**
     * List all e-mail generated in the specified period by optionally filtering
     * for specific senders and/or recipients.
     *
     * @param senderAddr The RFC822 address of the sender in question or <code>null</code>,
     *              if the result should not be filtered by sender.
     * @param recipientAddr The RFC822 address of the recipient in question or <code>null</code>,
     *              if the result should not be filtered by recipient.
     * @param from The date to find emails created after or <code>null</code>, if searching
     *              without lower bound.
     * @param to The date to find emails created before or <code>null</code>, if searching
     *              without upper bound.
     * @return A list of e-mails created in the requested period.
     */
    List<EMailReportDTO> getAllEMails(Date from, Date to, String senderAddr, String recipientAddr);

}
