package org.clazzes.fancymail.aws;

import java.io.InterruptedIOException;
import java.util.HashMap;
import java.util.Map;

import org.clazzes.fancymail.sms.ISMS;
import org.clazzes.fancymail.sms.ISMSChannel;
import org.clazzes.fancymail.sms.SMSException;
import org.clazzes.http.aws.JsonHelper;
import org.clazzes.http.aws.sns.AwsSNSService;
import org.clazzes.http.aws.sns.SNSMessageAttribute;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AWSSMSChannel implements ISMSChannel {

    private static final Logger log = LoggerFactory.getLogger(AWSSMSChannel.class);

    private final AwsSNSService snsService;

    private String senderID;
    private String maxPrice;
    private String smsType;
    private boolean suppressOriginationNumber;

    public AWSSMSChannel() {

        this.snsService= new AwsSNSService();
    }

    @Override
    public boolean sendSMSNow(ISMS sms) throws SMSException, InterruptedException {

        Map<SNSMessageAttribute,String> messageAttributes = new HashMap<SNSMessageAttribute,String>();

        if (sms.getSenderNumber() != null && !this.suppressOriginationNumber) {
            messageAttributes.put(SNSMessageAttribute.SMSOriginationNumber,
                                  sms.getSenderNumber());
        }

        if (this.senderID != null && !this.senderID.isEmpty()) {
            messageAttributes.put(SNSMessageAttribute.SMSSenderID,
                                  this.senderID);
        }

        if (this.smsType != null && !this.smsType.isEmpty()) {
            messageAttributes.put(SNSMessageAttribute.SMSType,
                                  this.smsType);
        }

        if (this.maxPrice != null && !this.maxPrice.isEmpty()) {
            messageAttributes.put(SNSMessageAttribute.SMSMaxPrice,
                                  this.maxPrice);
        }

        boolean success = true;
        boolean ret = true;

        StringBuilder sb = new StringBuilder();
        Exception exptn = null;

        for (String phone : sms.getRecipientNumbers()) {

            try {

                log.info("Publishing SMS to [{}] using AWS APIs...",phone);

                Map<String, ?> resp = this.snsService.publish(phone,sms.getText(),messageAttributes);

                log.info("AWS accepted SMS to [{}] with repsonse [{}].",
                         phone,resp);

                if (sb.length() > 0) {
                    sb.append(";");
                }

                String messageId =
                    JsonHelper.stringFromObjectPath(resp,"PublishResponse.PublishResult.MessageId");

                sb.append(phone+"/"+messageId);

                sms.setSent(sb.toString());
            }
            catch(Exception e) {

                log.error("AWS failed for SMS to ["+phone+"]",e);

                if (sb.length() > 0) {
                    sb.append(";");
                }

                sb.append(phone+"/"+e.getClass().getSimpleName());
                success = false;

                exptn = e;

                if ((e instanceof InterruptedException) || (e instanceof InterruptedIOException)) {
                    ret = false;
                }

                break;
            }
        }

        if (success) {
            sms.setSent(sb.toString());
        }
        else {
            sms.setUnsent(sb.toString(),exptn);
        }
        return ret;
    }

    @Override
    public boolean supportsSenderFaking() {

        return true;
    }

    public void setSenderID(String senderID) {
        this.senderID = senderID;
    }

    public void setMaxPrice(String maxPrice) {
        this.maxPrice = maxPrice;
    }

    public void setSmsType(String smsType) {
        this.smsType = smsType;
    }

    public void setSuppressOriginationNumber(boolean suppressOriginationNumber) {
        this.suppressOriginationNumber = suppressOriginationNumber;
    }

    public void setTimeoutSecs(int timeoutSecs) {
        this.snsService.setTimeoutSecs(timeoutSecs);
    }

    @Override
    public String toString() {
        return "AWSSMSChannel [senderID=" + this.senderID +
               ", maxPrice=" + this.maxPrice +
               ", smsType=" + this.smsType +
               ", suppressOriginationNumber=" + this.suppressOriginationNumber + "]";
    }


}
